"""Image processor class for ConvNeXT."""from typing import Dict, List, Optional, Unionimport numpy as npfrom ...image_processing_utils import BaseImageProcessor, BatchFeature, get_size_dictfrom ...image_transforms import (    center_crop,    get_resize_output_image_size,    resize,    to_channel_dimension_format,)from ...image_utils import (    IMAGENET_STANDARD_MEAN,    IMAGENET_STANDARD_STD,    ChannelDimension,    ImageInput,    PILImageResampling,    infer_channel_dimension_format,    is_scaled_image,    make_list_of_images,    to_numpy_array,    valid_images,)from ...utils import TensorType, is_vision_available, loggingif is_vision_available():    import PILlogger = logging.get_logger(__name__)class ConvNextImageProcessor(BaseImageProcessor):    r"""    Constructs a ConvNeXT image processor.    Args:        do_resize (`bool`, *optional*, defaults to `True`):            Controls whether to resize the image's (height, width) dimensions to the specified `size`. Can be overriden            by `do_resize` in the `preprocess` method.        size (`Dict[str, int]` *optional*, defaults to `{"shortest_edge": 384}`):            Resolution of the output image after `resize` is applied. If `size["shortest_edge"]` >= 384, the image is            resized to `(size["shortest_edge"], size["shortest_edge"])`. Otherwise, the smaller edge of the image will            be matched to `int(size["shortest_edge"]/crop_pct)`, after which the image is cropped to            `(size["shortest_edge"], size["shortest_edge"])`. Only has an effect if `do_resize` is set to `True`. Can            be overriden by `size` in the `preprocess` method.        crop_pct (`float` *optional*, defaults to 224 / 256):            Percentage of the image to crop. Only has an effect if `do_resize` is `True` and size < 384. Can be            overriden by `crop_pct` in the `preprocess` method.        resample (`PILImageResampling`, *optional*, defaults to `Resampling.BILINEAR`):            Resampling filter to use if resizing the image. Can be overriden by `resample` in the `preprocess` method.        do_rescale (`bool`, *optional*, defaults to `True`):            Whether to rescale the image by the specified scale `rescale_factor`. Can be overriden by `do_rescale` in            the `preprocess` method.        rescale_factor (`int` or `float`, *optional*, defaults to `1/255`):            Scale factor to use if rescaling the image. Can be overriden by `rescale_factor` in the `preprocess`            method.        do_normalize (`bool`, *optional*, defaults to `True`):            Whether to normalize the image. Can be overridden by the `do_normalize` parameter in the `preprocess`            method.        image_mean (`float` or `List[float]`, *optional*, defaults to `IMAGENET_STANDARD_MEAN`):            Mean to use if normalizing the image. This is a float or list of floats the length of the number of            channels in the image. Can be overridden by the `image_mean` parameter in the `preprocess` method.        image_std (`float` or `List[float]`, *optional*, defaults to `IMAGENET_STANDARD_STD`):            Standard deviation to use if normalizing the image. This is a float or list of floats the length of the            number of channels in the image. Can be overridden by the `image_std` parameter in the `preprocess` method.    """    model_input_names = ["pixel_values"]    def __init__(        self,        do_resize: bool = True,        size: Dict[str, int] = None,        crop_pct: float = None,        resample: PILImageResampling = PILImageResampling.BILINEAR,        do_rescale: bool = True,        rescale_factor: Union[int, float] = 1 / 255,        do_normalize: bool = True,        image_mean: Optional[Union[float, List[float]]] = None,        image_std: Optional[Union[float, List[float]]] = None,        **kwargs,    ) -> None:        super().__init__(**kwargs)        size = size if size is not None else {"shortest_edge": 384}        size = get_size_dict(size, default_to_square=False)        self.do_resize = do_resize        self.size = size        # Default value set here for backwards compatibility where the value in config is None        self.crop_pct = crop_pct if crop_pct is not None else 224 / 256        self.resample = resample        self.do_rescale = do_rescale        self.rescale_factor = rescale_factor        self.do_normalize = do_normalize        self.image_mean = image_mean if image_mean is not None else IMAGENET_STANDARD_MEAN        self.image_std = image_std if image_std is not None else IMAGENET_STANDARD_STD    def resize(        self,        image: np.ndarray,        size: Dict[str, int],        crop_pct: float,        resample: PILImageResampling = PILImageResampling.BICUBIC,        data_format: Optional[Union[str, ChannelDimension]] = None,        input_data_format: Optional[Union[str, ChannelDimension]] = None,        **kwargs,    ) -> np.ndarray:        """        Resize an image.        Args:            image (`np.ndarray`):                Image to resize.            size (`Dict[str, int]`):                Dictionary of the form `{"shortest_edge": int}`, specifying the size of the output image. If                `size["shortest_edge"]` >= 384 image is resized to `(size["shortest_edge"], size["shortest_edge"])`.                Otherwise, the smaller edge of the image will be matched to `int(size["shortest_edge"] / crop_pct)`,                after which the image is cropped to `(size["shortest_edge"], size["shortest_edge"])`.            crop_pct (`float`):                Percentage of the image to crop. Only has an effect if size < 384.            resample (`PILImageResampling`, *optional*, defaults to `PILImageResampling.BICUBIC`):                Resampling filter to use when resizing the image.            data_format (`str` or `ChannelDimension`, *optional*):                The channel dimension format of the image. If not provided, it will be the same as the input image.            input_data_format (`ChannelDimension` or `str`, *optional*):                The channel dimension format of the input image. If not provided, it will be inferred from the input                image.        """        size = get_size_dict(size, default_to_square=False)        if "shortest_edge" not in size:            raise ValueError(f"Size dictionary must contain 'shortest_edge' key. Got {size.keys()}")        shortest_edge = size["shortest_edge"]        if shortest_edge < 384:            # maintain same ratio, resizing shortest edge to shortest_edge/crop_pct            resize_shortest_edge = int(shortest_edge / crop_pct)            resize_size = get_resize_output_image_size(                image, size=resize_shortest_edge, default_to_square=False, input_data_format=input_data_format            )            image = resize(                image=image,                size=resize_size,                resample=resample,                data_format=data_format,                input_data_format=input_data_format,                **kwargs,            )            # then crop to (shortest_edge, shortest_edge)            return center_crop(                image=image,                size=(shortest_edge, shortest_edge),                data_format=data_format,                input_data_format=input_data_format,                **kwargs,            )        else:            # warping (no cropping) when evaluated at 384 or larger            return resize(                image,                size=(shortest_edge, shortest_edge),                resample=resample,                data_format=data_format,                input_data_format=input_data_format,                **kwargs,            )    def preprocess(        self,        images: ImageInput,        do_resize: bool = None,        size: Dict[str, int] = None,        crop_pct: float = None,        resample: PILImageResampling = None,        do_rescale: bool = None,        rescale_factor: float = None,        do_normalize: bool = None,        image_mean: Optional[Union[float, List[float]]] = None,        image_std: Optional[Union[float, List[float]]] = None,        return_tensors: Optional[Union[str, TensorType]] = None,        data_format: ChannelDimension = ChannelDimension.FIRST,        input_data_format: Optional[Union[str, ChannelDimension]] = None,        **kwargs,    ) -> PIL.Image.Image:        """        Preprocess an image or batch of images.        Args:            images (`ImageInput`):                Image to preprocess. Expects a single or batch of images with pixel values ranging from 0 to 255. If                passing in images with pixel values between 0 and 1, set `do_rescale=False`.            do_resize (`bool`, *optional*, defaults to `self.do_resize`):                Whether to resize the image.            size (`Dict[str, int]`, *optional*, defaults to `self.size`):                Size of the output image after `resize` has been applied. If `size["shortest_edge"]` >= 384, the image                is resized to `(size["shortest_edge"], size["shortest_edge"])`. Otherwise, the smaller edge of the                image will be matched to `int(size["shortest_edge"]/ crop_pct)`, after which the image is cropped to                `(size["shortest_edge"], size["shortest_edge"])`. Only has an effect if `do_resize` is set to `True`.            crop_pct (`float`, *optional*, defaults to `self.crop_pct`):                Percentage of the image to crop if size < 384.            resample (`int`, *optional*, defaults to `self.resample`):                Resampling filter to use if resizing the image. This can be one of `PILImageResampling`, filters. Only                has an effect if `do_resize` is set to `True`.            do_rescale (`bool`, *optional*, defaults to `self.do_rescale`):                Whether to rescale the image values between [0 - 1].            rescale_factor (`float`, *optional*, defaults to `self.rescale_factor`):                Rescale factor to rescale the image by if `do_rescale` is set to `True`.            do_normalize (`bool`, *optional*, defaults to `self.do_normalize`):                Whether to normalize the image.            image_mean (`float` or `List[float]`, *optional*, defaults to `self.image_mean`):                Image mean.            image_std (`float` or `List[float]`, *optional*, defaults to `self.image_std`):                Image standard deviation.            return_tensors (`str` or `TensorType`, *optional*):                The type of tensors to return. Can be one of:                    - Unset: Return a list of `np.ndarray`.                    - `TensorType.TENSORFLOW` or `'tf'`: Return a batch of type `tf.Tensor`.                    - `TensorType.PYTORCH` or `'pt'`: Return a batch of type `torch.Tensor`.                    - `TensorType.NUMPY` or `'np'`: Return a batch of type `np.ndarray`.                    - `TensorType.JAX` or `'jax'`: Return a batch of type `jax.numpy.ndarray`.            data_format (`ChannelDimension` or `str`, *optional*, defaults to `ChannelDimension.FIRST`):                The channel dimension format for the output image. Can be one of:                - `"channels_first"` or `ChannelDimension.FIRST`: image in (num_channels, height, width) format.                - `"channels_last"` or `ChannelDimension.LAST`: image in (height, width, num_channels) format.                - Unset: Use the channel dimension format of the input image.            input_data_format (`ChannelDimension` or `str`, *optional*):                The channel dimension format for the input image. If unset, the channel dimension format is inferred                from the input image. Can be one of:                - `"channels_first"` or `ChannelDimension.FIRST`: image in (num_channels, height, width) format.                - `"channels_last"` or `ChannelDimension.LAST`: image in (height, width, num_channels) format.                - `"none"` or `ChannelDimension.NONE`: image in (height, width) format.        """        do_resize = do_resize if do_resize is not None else self.do_resize        crop_pct = crop_pct if crop_pct is not None else self.crop_pct        resample = resample if resample is not None else self.resample        do_rescale = do_rescale if do_rescale is not None else self.do_rescale        rescale_factor = rescale_factor if rescale_factor is not None else self.rescale_factor        do_normalize = do_normalize if do_normalize is not None else self.do_normalize        image_mean = image_mean if image_mean is not None else self.image_mean        image_std = image_std if image_std is not None else self.image_std        size = size if size is not None else self.size        size = get_size_dict(size, default_to_square=False)        images = make_list_of_images(images)        if not valid_images(images):            raise ValueError(                "Invalid image type. Must be of type PIL.Image.Image, numpy.ndarray, "                "torch.Tensor, tf.Tensor or jax.ndarray."            )        if do_resize and size is None or resample is None:            raise ValueError("Size and resample must be specified if do_resize is True.")        if do_resize and size["shortest_edge"] < 384 and crop_pct is None:            raise ValueError("crop_pct must be specified if size < 384.")        if do_rescale and rescale_factor is None:            raise ValueError("Rescale factor must be specified if do_rescale is True.")        if do_normalize and (image_mean is None or image_std is None):            raise ValueError("Image mean and std must be specified if do_normalize is True.")        # All transformations expect numpy arrays.        images = [to_numpy_array(image) for image in images]        if is_scaled_image(images[0]) and do_rescale:            logger.warning_once(                "It looks like you are trying to rescale already rescaled images. If the input"                " images have pixel values between 0 and 1, set `do_rescale=False` to avoid rescaling them again."            )        if input_data_format is None:            # We assume that all images have the same channel dimension format.            input_data_format = infer_channel_dimension_format(images[0])        if do_resize:            images = [                self.resize(                    image=image, size=size, crop_pct=crop_pct, resample=resample, input_data_format=input_data_format                )                for image in images            ]        if do_rescale:            images = [                self.rescale(image=image, scale=rescale_factor, input_data_format=input_data_format)                for image in images            ]        if do_normalize:            images = [                self.normalize(image=image, mean=image_mean, std=image_std, input_data_format=input_data_format)                for image in images            ]        images = [            to_channel_dimension_format(image, data_format, input_channel_dim=input_data_format) for image in images        ]        data = {"pixel_values": images}        return BatchFeature(data=data, tensor_type=return_tensors)