
import asyncio
import atexit
import tortoise.contrib.fastapi
from fastapi import FastAPI
from fastapi import Request
from fastapi import status
from fastapi.middleware.cors import CORSMiddleware
from fastapi.responses import FileResponse
from fastapi.responses import JSONResponse
from fastapi.staticfiles import StaticFiles
from fastapi_restful.tasks import repeat_every
from pathlib import Path

from app import logging
from app.config import Config
from app.config import LoadConfig
from app.constants import (
    CLIENT_DIR,
    DATABASE_CONFIG,
    QUALITY,
    VERSION,
)
from app.models.Channel import Channel
from app.models.Program import Program
from app.models.TwitterAccount import TwitterAccount
from app.routers import (
    CapturesRouter,
    ChannelsRouter,
    DataBroadcastingRouter,
    LiveStreamsRouter,
    MaintenanceRouter,
    NiconicoRouter,
    ProgramsRouter,
    ReservesRouter,
    ReserveConditionsRouter,
    SeriesRouter,
    SettingsRouter,
    TwitterRouter,
    UsersRouter,
    VersionRouter,
    VideosRouter,
)
from app.routers import VideoStreamsRouter
from app.streams.LiveStream import LiveStream
from app.utils.EDCB import EDCBTuner


# もし Config() の実行時に AssertionError が発生した場合は、LoadConfig() を実行してサーバー設定データをロードする
## 自動リロードモードでは app.py がサーバープロセスのエントリーポイントになるため、
## サーバープロセス上にサーバー設定データがロードされていない状態になる
try:
    CONFIG = Config()
except AssertionError:
    # バリデーションは既にサーバー起動時に行われているためスキップする
    CONFIG = LoadConfig(bypass_validation=True)

# FastAPI を初期化
app = FastAPI(
    title = 'KonomiTV',
    description = 'KonomiTV: Kept Organized, Notably Optimized, Modern Interface TV media server',
    openapi_url = '/api/openapi.json',
    docs_url = '/api/docs',
    redoc_url = '/api/redoc',
    version = VERSION,
)

# ルーターの追加
app.include_router(ChannelsRouter.router)
app.include_router(ProgramsRouter.router)
app.include_router(VideosRouter.router)
app.include_router(SeriesRouter.router)
app.include_router(LiveStreamsRouter.router)
app.include_router(VideoStreamsRouter.router)
app.include_router(ReservesRouter.router)
app.include_router(ReserveConditionsRouter.router)
app.include_router(CapturesRouter.router)
app.include_router(DataBroadcastingRouter.router)
app.include_router(NiconicoRouter.router)
app.include_router(TwitterRouter.router)
app.include_router(UsersRouter.router)
app.include_router(SettingsRouter.router)
app.include_router(MaintenanceRouter.router)
app.include_router(VersionRouter.router)

# CORS の設定
app.add_middleware(
    CORSMiddleware,
    allow_origins = [
        # デバッグモード時のみ CORS ヘッダーを有効化 (クライアント側の開発サーバーからのアクセスに必要)
        '*' if CONFIG.general.debug is True else '',
    ],
    allow_methods = ["*"],
    allow_headers = ["*"],
    allow_credentials = True,
)

# 静的ファイルの配信
app.mount('/assets', StaticFiles(directory=CLIENT_DIR / 'assets', html=True))

# ルート以下のルーティング
# ファイルが存在すればそのまま配信し、ファイルが存在しなければ index.html を返す
@app.get('/{file:path}', include_in_schema=False)
async def Root(file: str):

    # ディレクトリトラバーサル対策のためのチェック
    ## ref: https://stackoverflow.com/a/45190125/17124142
    try:
        CLIENT_DIR.joinpath(Path(file)).resolve().relative_to(CLIENT_DIR.resolve())
    except ValueError:
        # URL に指定されたファイルパスが CLIENT_DIR の外側のフォルダを指している場合は、
        # ファイルが存在するかに関わらず一律で index.html を返す
        return FileResponse(CLIENT_DIR / 'index.html', media_type='text/html')

    # ファイルが存在する場合のみそのまま配信
    filepath = CLIENT_DIR / file
    if filepath.is_file():
        # 拡張子から MIME タイプを判定
        if filepath.suffix == '.css':
            mime = 'text/css'
        elif filepath.suffix == '.html':
            mime = 'text/html'
        elif filepath.suffix == '.ico':
            mime = 'image/x-icon'
        elif filepath.suffix == '.js':
            mime = 'application/javascript'
        elif filepath.suffix == '.json':
            mime = 'application/json'
        elif filepath.suffix == '.map':
            mime = 'application/json'
        else:
            mime = 'text/plain'
        return FileResponse(filepath, media_type=mime)

    # デフォルトドキュメント (index.html)
    # URL の末尾にスラッシュがついている場合のみ
    elif (filepath / 'index.html').is_file() and (file == '' or file[-1] == '/'):
        return FileResponse(filepath / 'index.html', media_type='text/html')

    # 存在しない静的ファイルが指定された場合
    else:
        if file.startswith('api/'):
            # パスに api/ が前方一致で含まれているなら、404 Not Found を返す
            return JSONResponse({'detail': 'Not Found'}, status_code = status.HTTP_404_NOT_FOUND)
        else:
            # パスに api/ が前方一致で含まれていなければ、index.html を返す
            return FileResponse(CLIENT_DIR / 'index.html', media_type='text/html')

# Internal Server Error のハンドリング
@app.exception_handler(Exception)
async def ExceptionHandler(request: Request, exc: Exception):
    return JSONResponse(
        {'detail': f'Oops! {type(exc).__name__} did something. There goes a rainbow...'},
        status_code = status.HTTP_500_INTERNAL_SERVER_ERROR,
        # FastAPI の謎仕様で CORSMiddleware は exception_handler に対しては効かないので、ここで自前で CORS ヘッダーを付与する
        headers = {'Access-Control-Allow-Origin': '*'},
    )

# Tortoise ORM の初期化
## ロガーを Uvicorn に統合する
## ref: https://github.com/tortoise/tortoise-orm/issues/529
tortoise.contrib.fastapi.logging = logging.logger  # type: ignore
## Tortoise ORM を登録する
## ref: https://tortoise-orm.readthedocs.io/en/latest/contrib/fastapi.html
tortoise.contrib.fastapi.register_tortoise(
    app = app,
    config = DATABASE_CONFIG,
    generate_schemas = True,
    add_exception_handlers = True,
)

# サーバーの起動時に実行する
@app.on_event('startup')
async def Startup():

    # チャンネル情報を更新
    await Channel.update()

    # ニコニコ実況関連のステータスを更新
    await Channel.updateJikkyoStatus()

    # 番組情報を更新
    await Program.update()

    # 登録されている Twitter アカウントの情報を更新
    await TwitterAccount.updateAccountsInformation()

    # 全てのチャンネル&品質のライブストリームを初期化する
    for channel in await Channel.filter(is_watchable=True).order_by('channel_number'):
        for quality in QUALITY:
            LiveStream(channel.display_channel_id, quality)

    # 録画フォルダ配下の録画ファイルのメタデータを更新/同期
    ## 録画ファイルの量次第では録画ファイルの更新確認に時間がかかるため、非同期で実行する
    async def run():
        # サーバーの起動完了を待ってから実行する
        await asyncio.sleep(0.1)
        # await RecordedVideo.update()
    asyncio.create_task(run())

# サーバー設定で指定された時間 (デフォルト: 15分) ごとに1回、チャンネル情報と番組情報を更新する
# チャンネル情報は頻繁に変わるわけではないけど、手動で再起動しなくても自動で変更が適用されてほしい
# 番組情報の更新処理はかなり重くストリーム配信などの他の処理に影響してしまうため、マルチプロセスで実行する
@app.on_event('startup')
@repeat_every(
    seconds = CONFIG.general.program_update_interval * 60,
    wait_first = CONFIG.general.program_update_interval * 60,
    logger = logging.logger,
)
async def UpdateChannelAndProgram():
    await Channel.update()
    await Channel.updateJikkyoStatus()
    await Program.update(multiprocess=True)

# 30秒に1回、ニコニコ実況関連のステータスを更新する
@app.on_event('startup')
@repeat_every(seconds=0.5 * 60, wait_first=0.5 * 60, logger=logging.logger)
async def UpdateChannelJikkyoStatus():
    await Channel.updateJikkyoStatus()

# 1時間に1回、登録されている Twitter アカウントの情報を更新する
@app.on_event('startup')
@repeat_every(seconds=60 * 60, wait_first=60 * 60, logger=logging.logger)
async def UpdateTwitterAccountInformation():
    await TwitterAccount.updateAccountsInformation()

# サーバーの終了時に実行する
cleanup = False
@app.on_event('shutdown')
async def Shutdown():

    # 2度呼ばれないように
    global cleanup
    if cleanup is True:
        return
    cleanup = True

    # 全てのライブストリームを終了する
    for live_stream in LiveStream.getAllLiveStreams():
        live_stream.setStatus('Offline', 'ライブストリームは Offline です。', True)

    # 全てのチューナーインスタンスを終了する (EDCB バックエンドのみ)
    if CONFIG.general.backend == 'EDCB':
        await EDCBTuner.closeAll()

# shutdown イベントが発火しない場合も想定し、アプリケーションの終了時に Shutdown() が確実に呼ばれるように
# atexit は同期関数しか実行できないので、asyncio.run() でくるむ
atexit.register(asyncio.run, Shutdown())
