import logging
import requests
import traceback

from alerter.common import Alerter, AlerterFactory


@AlerterFactory.register
class SlackAlerter(Alerter):
    def __init__(self, **kwargs):
        super().__init__(**kwargs)
        self.webhook_url = kwargs.get('webhook_url')
        self.mentions = kwargs.get('mentions', '')
        if self.mentions:
            self.mentions = ' '.join([f'<@{m}>' for m in self.mentions])

    @classmethod
    def from_args(cls, args):
        webhook_url = args.webhook_url
        return cls(webhook_url=webhook_url)

    @classmethod
    def from_config(cls, config):
        webhook_url = config['webhook_url']
        mentions = config['mentions'] if 'mentions' in config else ''
        return cls(webhook_url=webhook_url, mentions=mentions)

    @staticmethod
    def get_alerter_type():
        return 'slack'

    def __call__(self, **kwargs):
        _slack_webhook_generated = {
            "blocks": [
                {
                    "type": "section",
                    "text": {
                        "type": "mrkdwn",
                        "text": "*Inventory Hunter* :mega:"
                    }
                },
                {
                    "type": "section",
                    "text": {
                        "type": "mrkdwn",
                        "text": kwargs.get("content")
                    }
                }
            ]
        }

        if self.mentions:
            blocks = _slack_webhook_generated["blocks"]
            mentions = {
                "type": "section",
                "text": {
                        "type": "mrkdwn",
                        "text": self.mentions
                }
            }
            blocks.append(mentions)

        try:
            logging.debug(f"Slack Webhook URL: {self.webhook_url}")
            send_request = requests.post(
                self.webhook_url,
                json=_slack_webhook_generated,
            )
            if send_request.status_code != 200:
                logging.error(
                    f"There was an issue sending to slack due to an invalid request: {send_request.status_code} -> {send_request.text}"
                )
        except Exception:
            logging.error(
                f"Issue with sending webhook to slack. {traceback.format_exc()}"
            )
