#
# An example of a VD for CWE252
# In this example, we do not have a CVE.
# But we use the ping6 binary from inetutils package.
# This bug was discovered during the evaluation of Arbiter
# The vulnerability has now been fixed.
# https://www.mail-archive.com/bug-inetutils@gnu.org/msg03149.html
#
# The vulnerability is an unchecked call to setuid in the main function
# inside ping.c and has been fixed in commit 3ce348e63c3934958a2665ef8289d28a12150948
#
# int main (int argc, char **argv) {
# ...
#
# setuid (getuid ());
#
# ...
# }
#
# The VD that we can generate from this information is as follows
# Sink := End of function containing `setuid`
# Source := Return value of `setuid`
# Constraint := Return value of `setuid` must be constrained to either -1 or 0, but never satisfy both together.
#
# This CWE type is different from others because of the interesting constraint used.
# To elaborate, imagine the following code.
#
# ...
# int x = critical_function();
# if (x == ERROR_CODE)
#     return_value = -1;                // A
# else
#     return_value = 0;                 // B
#
# return return_value;                  // C
# ...
#
# In this case, a state that reaches C could have either executed statement A or statement B.
# Without more context, we cannot know whether this state is correctly handling a success/error case.
# All we can say at C is : `x` == ERROR_CODE if A was executed or `x` != ERROR_CODE if B was executed.
# Therefore a single state cannot satisfy the constraints `x` == ERROR_CODE and `x` != ERROR_CODE
#
# However, if the state does not check the return value, there will be no constraints on the return value.
# In, such a state, the return value can be either success or error.
# If we detect such a state, we can be sure (to an extent) that the error case is not handled at all.
#


def apply_constraint(state, expr, init_val, **kwargs):
    #
    # Here, `expr` represents the return value of `main`.
    # The return value of `setuid` will be present inside `init_val`
    #
    #
    # Our objective is to find a situation where one single state can satisfy error and success conditions.
    # Ideally, the best way to do this would be to add one constraint and check if its true.
    # Then, remove this first constraint and add the other and repeat the check.
    # However, removing the constraint from the state does not seem to remove it completely.
    # Therefore, we create a copy of the state.
    #
    s1 = state.copy()

    #
    # First we add the error constraint.
    # In this case, a return value of -1 indicates error.
    # `setuid` returns a 32 bit integer
    # So -1 in 32 bit is 0xFFFFFFFF
    #
    s1.solver.add(init_val[0] == 0xFFFFFFFF)
    if s1.satisfiable():
        #
        # Now, we add the success constraint.
        # In this case, a return value of 0 indicates success.
        #
        state.solver.add(init_val[0] == 0)
    else:
        #
        state.solver.add(init_val[0] == 0xFFFFFFFF)
    return


def specify_sources():
    #
    # The source we're interested in is the return value of `setuid`
    #
    return {'setuid': 0}


def specify_sinks():
    #
    # Every function contains a ret instruction.
    # In order to filter it down, we focus on functions that contain the source.
    # Use 'r' to denote that we're interested in the ret instruction of the function containing `setuid`.
    #

    maps = {'setuid': ['r']}
    return maps


def save_results(reports):
    return